// load92
//
// Loads data from a SPG file into a PRO-92 scanner.
//
// Turbo version - does the upload in 1:15, the same as the clone time.
//
// Written by Ken Plotkin
//            kjp15@cornell.edu
//            December 2000
//            February 2000
//
// Translated from Fortran to C by Steve Falco
// 				   sfalco@worldnet.att.net
// 				   September 2001
// 				   This version is "Unix-centric" in that it
// 				   depends on various Unix system calls to
// 				   establish the parameters of the serial
// 				   connection.  Debugged and tested on RedHat
// 				   Linux version 7.1.
//
// Distributed as careware.  If you like it, donate some money to a worthwhile
// charity.

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <time.h>

typedef unsigned char UCHAR;
typedef unsigned int UINT;

#define SPLEN	50	// Scanner pre-pad length
#define FHLEN	5	// Header stored in disk file
#define BLEN	26496	// Body length
#define MBLEN	30000	// Body length (plus some margin)
#define EMLEN	(SPLEN + FHLEN + BLEN)	// Location of end marker
#define PSB1	(SPLEN + FHLEN - 1)	// First parity switch byte
#define PSB2	(SPLEN + FHLEN + BLEN - 1) // Second parity switch byte
#define SNOOZE	20	// Default snooze time in milliseconds
#define IBLK	32	// Do not change - it is fixed to the ouput rhythm of
			// the scanner.
#define EOL	"\n"	// end of line

char	*infile = "load.spg";
char	*port = "/dev/ttyS0";

struct termios new_settings;

void use();

main(int argc, char *argv[])
{
	extern char *optarg;
	extern int optind, opterr, optopt;

	UCHAR	clndat[MBLEN];
	UCHAR	revdat[MBLEN];
	UCHAR	errbuf[BUFSIZ];

	UCHAR	*dot;
	UCHAR	flag;

	int	infd, outfd, ic, i, isnooz, nblok;

	time_t	starttime, delta;

	starttime = time(0);

	// Inter-iblk delay is normally 20 msec. -snnn changes it
	isnooz = SNOOZE;
	while((ic = getopt(argc, argv, "hp:s:")) != -1) {
		switch(ic) {
			case 's':
				isnooz = atoi(optarg);
				break;
			case 'p':
				port = optarg;
				break;
			case 'h': /*FALLTHROUGH*/
			case '?':
				use();
				break;
		}
	}


	if(optind < argc) {
		infile = argv[optind];
	}

	// Keep the user out of trouble - part 1
	if( !(					     // negate the whole thing
		((dot = strrchr(infile, '.')) != 0) // we found the last dot
		&&				     // and
		( 				     // (
		    (strcmp(dot, ".spg") == 0)	     // we have .spg
		    ||				     // or
		    (strcmp(dot, ".SPG") == 0)	     // we have .spg
		)				     // )
	)) {
		fprintf(stderr, "File name (%s) must end with .spg" EOL, infile);
		exit(1);
	}

	// Open the file containing the data stream.  It gets placed into
	// array clndat, which looks just like a clone dump (without a
	// leading CD).
	if((infd = open(infile, O_RDONLY)) == -1) {
		sprintf(errbuf, "Cannot open %s", infile);
		perror(errbuf);
		exit(1);
	}

	// Set up the data start and data end bytes
	memset(clndat + 1, 0xA5, SPLEN);
	clndat[SPLEN + 1] = 0x01;
	clndat[SPLEN + 2] = 0x98;
	clndat[SPLEN + 3] = 0x03;
	clndat[SPLEN + 4] = 0x52;
	clndat[EMLEN]     = 0x5A;

	// Ignore the first FHLEN bytes
	if(lseek(infd, FHLEN, SEEK_SET) == -1) {
		sprintf(errbuf, "Cannot seek past header of %s", port);
		perror(errbuf);
		exit(1);
	}

	// The data are stored in reverse order.  Read the data into our
	// reversing array.
	if(read(infd, revdat, BLEN) != BLEN) {
		sprintf(errbuf, "Cannot read spg data from %s", infile);
		perror(errbuf);
		exit(1);
	}

	// Now reverse the array into our output buffer
	for(i = 0; i < BLEN; i++) {
		clndat[EMLEN - 1 - i] = revdat[i];
	}

	// Open serial port to scanner.  Parameters are:
	// 	4800 baud
	// 	8 data bits
	// 	2 stop bits
	// 	even parity
	if((outfd = open(port, O_RDWR)) == -1) {
		sprintf(errbuf, "Cannot open %s", port);
		perror(errbuf);
		exit(1);
	}
	new_settings.c_iflag = 0;
	new_settings.c_oflag = 0;
	new_settings.c_cflag = 0;
	new_settings.c_lflag = 0;
	if(cfsetospeed(&new_settings, B4800) == -1) {
		sprintf(errbuf, "Cannot set baud rate for %s", port);
		perror(errbuf);
		exit(1);
	}
	new_settings.c_cflag |= CREAD;		// enable input
	new_settings.c_cflag |= CLOCAL;		// ignore all control lines
	new_settings.c_cflag |= CS8;		// want 8 data bits
	new_settings.c_cflag |= CSTOPB;		// want 2 stop bits
	new_settings.c_cflag |= PARENB;		// parity is enabled
	new_settings.c_cflag &= ~PARODD;	// and even
	if(tcsetattr(outfd, TCSANOW, &new_settings) == -1) {
		sprintf(errbuf, "Cannot set parameters for %s", port);
		perror(errbuf);
		exit(1);
	}

	// Set DTR (+12 volts), clear RTS (-12 volts).  These pins power
	// the hardware interface.  For some reason, DTR has to be set before
	// RTS in order for this to work correctly.
	flag = TIOCM_DTR;
	if(ioctl(outfd, TIOCMBIS, &flag)  == -1) {
		sprintf(errbuf, "Cannot set DTR for %s", port);
		perror(errbuf);
		exit(1);
	}
	flag = TIOCM_RTS;
	if(ioctl(outfd, TIOCMBIC, &flag)  == -1) {
		sprintf(errbuf, "Cannot clear RTS for %s", port);
		perror(errbuf);
		exit(1);
	}

	nblok = 0;
	for(i = 1; i <= EMLEN; i++) {
		// Put the next character into the output queue.
		if(write(outfd, &clndat[i], 1) != 1) {
			sprintf(errbuf, "Cannot write byte %d to %s", i, port);
			perror(errbuf);
			exit(1);
		}
		nblok++;

		// Every IBLK bytes, or when we reach PSB1 or PSB2 (where
		// parity is changed), pause and update the % done message.
		if((i >= PSB1)
		   && (
			(nblok == IBLK)
			|| (i == PSB1)
			|| (i >= PSB2)
		)) {
			if(tcdrain(outfd) == -1) {
				sprintf(errbuf, "Cannot drain output to %s", port);
				perror(errbuf);
				exit(1);
			}
			usleep(isnooz * 1000);

			// Extra sleep at PSB1, the end of the setup bytes (we
			// are already drained from above).
			if(i == PSB1) {
				usleep((isnooz * 2) * 1000);
			}

			// Print the percentage we have completed.  Note the
			// '\r' so we overwrite previous values.
			fprintf(stderr, "%3d%%\r", i/265);
			nblok = 0;
		}

		// The PSB1'th byte of the data stream put the scanner into
		// data receive mode.  Change parity to odd for the data
		// itself.
		if(i == PSB1) {
			if(tcdrain(outfd) == -1) {
				sprintf(errbuf, "Cannot drain output to %s", port);
				perror(errbuf);
				exit(1);
			}
			new_settings.c_cflag |= PARODD;// parity is odd
			if(tcsetattr(outfd, TCSANOW, &new_settings) == -1) {
				sprintf(errbuf, "Cannot flip to odd parity for %s", port);
				perror(errbuf);
				exit(1);
			}
		}

		// Byte PSB2 was the last of the data.  Change parity back
		// to even for the terminator.
		if(i == PSB2) {
			if(tcdrain(outfd) == -1) {
				sprintf(errbuf, "Cannot drain output to %s", port);
				perror(errbuf);
				exit(1);
			}
			new_settings.c_cflag &= ~PARODD;//parity is even
			if(tcsetattr(outfd, TCSANOW, &new_settings) == -1) {
				sprintf(errbuf, "Cannot flip to even parity for %s", port);
				perror(errbuf);
				exit(1);
			}
		}
	}
	fprintf(stderr, EOL);
	if(tcdrain(outfd) == -1) {
		sprintf(errbuf, "Cannot drain output to %s", port);
		perror(errbuf);
		exit(1);
	}
	delta = (int)(time(0) - starttime);
	fprintf(stderr, "Done!  Enjoy the upload.  (time was %02ld:%02ld)" EOL, delta / 60, delta % 60);

	close(infd);
	close(outfd);

	exit(0);
}

void
use()
{
	fprintf(stderr, "load92 [-h] [-p port] [-s snooze] [filename.spg]" EOL);
	fprintf(stderr, "  -h prints this help message" EOL);
	fprintf(stderr, "  -p port sets the device where the scanner is connected" EOL);
	fprintf(stderr, "          (typically /dev/ttyS0 or /dev/ttyS1," EOL);
	fprintf(stderr, "          defaults to %s)" EOL, port);
	fprintf(stderr, "  -s snooze sets the interblock delay" EOL);
	fprintf(stderr, "          (units are milliseconds, default is %d milliseconds)" EOL, SNOOZE);
	fprintf(stderr, "  filename.spg is the input file which must end with .spg" EOL);
	fprintf(stderr, "          (defaults to %s)" EOL, infile);

	exit(1);
}
